﻿using ArxOne.MrAdvice.Advice;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.Caching;
using System.Security.Cryptography;
using System.Text;
using System.Threading.Tasks;
using ConsoleApp2.Json;

namespace ConsoleApp2
{
    /// <summary>
    /// 用AOP来实现自动缓存
    /// </summary>
    public class AutoCacheAttribute : Attribute, IMethodAdvice
    {
        /// <summary>
        /// 滑动过期
        /// </summary>
        public bool EnableSliding { get; set; }

        /// <summary>
        /// 缓存时间，分钟
        /// </summary>
        public int CacheMinutes { get; set; }

        /// <summary>
        /// 构造函数
        /// </summary>
        /// <param name="cacheMinutes">缓存时间，分钟，默认5分钟，小于等于0永久缓存</param>
        /// <param name="enableSliding">使用滑动过期缓存控制策略</param>
        public AutoCacheAttribute(int cacheMinutes = 5, bool enableSliding = false)
        {
            EnableSliding = enableSliding;
            CacheMinutes = cacheMinutes;
        }

        /// <summary>
        /// AOP组件拦截方法，用于实现自动缓存，有缓存时直接返回；
        /// 没有缓存时，调用被拦截方法后，有返回值则将数据自动缓存起来
        /// </summary>
        /// <param name="context"></param>
        public void Advise(MethodAdviceContext context)
        {
            var key = GetKey(context);
            if (context.HasReturnValue && key.TryGetCache(out object m))
            {
                var r = m as ResponseResult;
                r.Message = "在拦截方法里面改了缓存里面取出来的数据！";

                context.ReturnValue = r;
                //context.ReturnValue = m;  

                //context.Proceed();  //直接取出缓存返回，不用执行原来取数据方法。
            }
            else
            {
                context.Proceed();//执行被拦截的方法
                if (context.HasReturnValue && context.ReturnValue != null)
                {
                    //被拦截方法有返回值，并且返回值不为null
                    if (EnableSliding && CacheMinutes > 0)
                        context.ReturnValue.SetCache(key, TimeSpan.FromMinutes(CacheMinutes));
                    else if (CacheMinutes > 0)
                        context.ReturnValue.SetCache(key, DateTime.Now.AddMinutes(CacheMinutes));
                    else
                        context.ReturnValue.SetCache(key);
                }
            }
        }

        /// <summary>
        /// 获取缓存key，key的规则为： md5(类全名｜方法名｜参数列表拆分数组｜参数值的json数组)，这样可以保证唯一
        /// </summary>
        /// <param name="context"></param>
        /// <returns></returns>
        private string GetKey(MethodAdviceContext context)
        {
            var array = context.TargetMethod.GetParameters();
            var key = array.Select(x => { return context.Arguments[x.Position].ToJson(); }).ToArray();

            var cacheKey = $"{context.Target.ToString()}|{context.TargetName}|{string.Join("_", array.Select(x => x.Name))}|{string.Join("_", key)}".GetMd5();
            return cacheKey;
        }
    }

    /// <summary>
    /// 缓存扩展方法，可使用其它缓存替代
    /// </summary>
    public static class CacheExtensions
    {
        private static MemoryCache cache = new MemoryCache("https://jhrs.com");

        /// <summary>
        /// 设置缓存，一直不过期
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="value"></param>
        /// <param name="key"></param>
        public static void SetCache<T>(this T value, string key)
        {
            if (string.IsNullOrWhiteSpace(key)) throw new ArgumentException($"缓存键参数{nameof(key)}不能为null或空");
            if (value == null) throw new ArgumentException($"缓存值参数{nameof(value)}不能为null");
            CacheItemPolicy policy = new CacheItemPolicy();
            cache.Set(key, value, policy);
        }

        /// <summary>
        /// 设置缓存，固定过期时间
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="value"></param>
        /// <param name="key"></param>
        /// <param name="absoluteExpiration"></param>
        public static void SetCache<T>(this T value, string key, DateTimeOffset? absoluteExpiration)
        {
            if (string.IsNullOrWhiteSpace(key)) throw new ArgumentException($"缓存键参数{nameof(key)}不能为null或空");
            if (value == null) throw new ArgumentException($"缓存值参数{nameof(value)}不能为null");
            CacheItemPolicy policy = new CacheItemPolicy() { AbsoluteExpiration = (DateTimeOffset)absoluteExpiration };
            cache.Set(key, value, policy);
        }

        /// <summary>
        /// 设置缓存，滑动过期
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="value"></param>
        /// <param name="key"></param>
        /// <param name="slidingExpiration"></param>
        public static void SetCache<T>(this T value, string key, TimeSpan? slidingExpiration)
        {
            if (string.IsNullOrWhiteSpace(key)) throw new ArgumentException($"缓存键参数{nameof(key)}不能为null或空");
            if (value == null) throw new ArgumentException($"缓存值参数{nameof(value)}不能为null");
            CacheItemPolicy policy = new CacheItemPolicy() { SlidingExpiration = (TimeSpan)slidingExpiration };
            cache.Set(key, value, policy);
        }

        /// <summary>
        /// 获取缓存数据
        /// </summary>
        /// <typeparam name="T">对象类型</typeparam>
        /// <param name="key"><缓存key/param>
        /// <param name="value">返回的缓存数据对名</param>
        /// <returns></returns>
        public static bool TryGetCache<T>(this string key, out T value)
        {
            value = default(T);
            if (cache.Contains(key))
            {
                value = (T)cache.Get(key);
                return true;
            }
            return false;
        }

        /// <summary>
        /// 获取字符串MD5值
        /// </summary>
        /// <param name="value"></param>
        /// <returns></returns>
        public static string GetMd5(this string value)
        {
            byte[] bytes = Encoding.UTF8.GetBytes(value);

            StringBuilder sb = new StringBuilder();
            MD5 hash = new MD5CryptoServiceProvider();
            bytes = hash.ComputeHash(bytes);
            foreach (byte b in bytes)
            {
                sb.AppendFormat("{0:x2}", b);
            }
            return sb.ToString();
        }
    }
}
